program ROMBERGMETHOD;
{--------------------------------------------------------------------}
{  Alg7'34.pas   Pascal program for implementing Algorithm 7.3-4     }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 7.3 (Recursive Trapezoidal Rule).                       }
{  Section   7.3, Recursive Rules and Romberg Integration, Page 378  }
{                                                                    }
{  Algorithm 7.4 (Romberg Integration).                              }
{  Section   7.3, Recursive Rules and Romberg Integration, Page 379  }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    FunMax = 9;
    MaxN = 12;
    Epsilon = 1E-9;

  type
    MATRIX = array[0..MaxN, 0..MaxN] of real;
    LETTER = string[1];
    LETTERS = string[200];
    States = (Changes, Done, Working);
    DoSome = (Go, Stop);

  var
    FunType, Inum, J, M, Sub: integer;
    A, B, Close, Rnum, Tol: real;
    R: MATRIX;
    Ans: CHAR;
    Mess: LETTERS;
    State: States;
    DoMo: DoSome;

  function F (var X: real): real;
  begin
    case FunType of
      0: 
        begin
          if X <> 0 then
            F := SIN(X) / X
          else
            F := 1;
        end;
      1: 
        begin
          if X <> 0 then
            F := 1 / X
          else
            begin
              if A = 0 then
                F := 1E37;
              if B = 0 then
                F := -1E37;
              if A * B < 0 then
                F := 0;
            end;
        end;
      2: 
        F := SIN(X);
      3: 
        F := 4 / (1 + X * X);
      4: 
        F := X / (1 + X * X);
      5: 
        begin
          if 2 * X <> 7 then
            F := 1 / (7 - 2 * X)
          else
            begin
              if A = 3.5 then
                F := -1E37;
              if B = 3.5 then
                F := 1E37;
              if (2 * A - 7) * (2 * B - 7) < 0 then
                F := 0;
            end;
        end;
      6: 
        begin
          if X = 3.5 then
            F := -1E37
          else
            F := LN(ABS(X));
        end;
      7: 
        F := X * EXP(-X);
      8: 
        F := 64 * (X - 1) * EXP(-X * LN(4));
      9: 
        F := EXP(-X * X / 2) / SQRT(2 * PI);
    end;
  end;

  procedure PRINTFUNCTION (FunType: integer);
  begin
    case FunType of
      0: 
        WRITE('SIN(X)/X');
      1: 
        WRITE('1/X');
      2: 
        WRITE('SIN(X)');
      3: 
        WRITE('4/(1+X^2)');
      4: 
        WRITE('X/(1+X^2)');
      5: 
        WRITE('1/(7-2*X)');
      6: 
        WRITE('LN|X|');
      7: 
        WRITE('X*EXP(-X)');
      8: 
        WRITE('64*(X-1)*4^-X');
      9: 
        WRITE('EXP(-X^2/2)/SQRT(2*Pi)');
    end;
  end;

  procedure ROMBERG ( {FUNCTION F(X:real): real;}
                  var A, B, Close: real; Tol: real; var J, M: integer);
    var
      K, Min, P: integer;
      H, SUM, X: real;

    function E4 (var J: integer): real;
      var                                                     {Compute 4^J}
        I: integer;
        P: real;
    begin
      P := 1;
      for I := 1 to J do
        P := P * 4;
      E4 := P;
    end;

    procedure TRAPEZOIDRULE;
      var
        P: integer;
    begin                               {The Sequential Trapezoidal Rule}
      H := H / 2;
      SUM := 0;
      for P := 1 to M do
        begin
          X := A + H * (2 * P - 1);
          SUM := SUM + F(X);
        end;
      R[J, 0] := H * SUM + R[J - 1, 0] / 2;
      M := 2 * M
    end;

  begin                            {The main part of Procedure Romberg}
    Min := 1;
    M := 1;
    H := B - A;
    Close := 1;
    J := 0;
    R[0, 0] := H * (F(A) + F(B)) / 2;
    while ((Close > Tol) and (J < MaxN)) or (J < Min) do
      begin                                {Richardson`s Extrapolation}
        J := J + 1;
        TRAPEZOIDRULE;
        for K := 1 to J do
          R[J, K] := R[J, K - 1] + (R[J, K - 1] - R[J - 1, K - 1]) / (E4(K) - 1);
        Close := ABS(R[J - 1, J - 1] - R[J, J]);
      end;
  end;

  procedure MESSAGE (var Tol: real);
  begin
    CLRSCR;
    WRITELN('           Romberg integration is performed to find an');
    WRITELN;
    WRITELN('     approximation for the value of the definite integral:');
    WRITELN;
    WRITELN('                   B');
    WRITELN('                   /');
    WRITELN('                   | f(x) dx  ~  R(J,J)');
    WRITELN('                   /');
    WRITELN('                   A');
    WRITELN;
    WRITELN('     Successive approximations are stored in the array  R(J,K).');
    WRITELN;
    WRITELN('                   R(J,0)');
    WRITELN('                   R(J,0)  R(J,1)');
    WRITELN('                   R(J,0)  R(J,1)  R(J,2)');
    WRITELN('                      .       .       .');
    WRITELN('                      :       :       :');
    WRITELN('                   R(J,0)  R(J,1)  R(J,2) ... R(J,J)');
    WRITELN;
    WRITELN('     The algorithm is terminated in the J-th row when');
    WRITELN;
    WRITELN('                   |R(J-1,J-1)-R(J,J)| < TOL');
    WRITELN;
    Mess := '                    ENTER  the  value    TOL = ';
    Tol := Epsilon;
    WRITE(Mess);
    READLN(Tol);
    Tol := ABS(Tol);
    if (Tol < Epsilon) then
      Tol := Epsilon;
  end;

  procedure INPUT (var A, B: real);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN('           Romberg integration is performed to find an');
    WRITELN;
    WRITELN('     approximation for the value of the definite integral:');
    WRITELN;
    WRITELN('                   B');
    WRITELN('                   /');
    WRITELN('                   | F(X) DX');
    WRITELN('                   /');
    WRITELN('                   A');
    WRITELN;
    WRITELN('     CHOOSE your function:');
    WRITELN;
    for K := 0 to FunMax do
      begin
        WRITE('             <', K : 2, ' >   F(X) = ');
        PRINTFUNCTION(K);
        WRITELN;
      end;
    WRITELN;
    WRITELN;
    WRITE('                     SELECT < 0 - ', FunMax : 1, ' > ?  ');
    FunType := 0;
    READLN(FunType);
    if FunType < 0 then
      FunType := 0;
    if FunType > FunMax then
      FunType := FunMax;
  end;

  procedure PROBLEM (FunType: integer);
  begin
    CLRSCR;
    WRITELN;
    WRITELN('     You chose to approximate the definite integral:');
    WRITELN;
    WRITELN('             B');
    WRITELN('             /');
    WRITE('             | ');
    PRINTFUNCTION(FunType);
    WRITELN(' DX');
    WRITELN('             /');
    WRITELN('             A');
    WRITELN;
  end;

  procedure EPOINTS (var A, B: real; var State: STATES);
    type
      STATUS = (Change, Enter, Done);
      LETTER = string[1];
    var
      Valu: real;
      Resp: CHAR;
      Stat: STATUS;
  begin
    Stat := Enter;
    if State = Changes then
      Stat := Change;
    while (Stat = Enter) or (Stat = Change) do
      begin
        PROBLEM(FunType);
        WRITELN;
        WRITELN('     The tolerance value is TOL =', Tol : 15 : 7);
        if (Stat = Enter) then
          begin
            WRITELN;
            WRITELN('     Now you must choose the interval [A,B].');
            ;
            Mess := '             ENTER  the left  endpoint A = ';
            A := 0;
            WRITE(Mess);
            READLN(A);
            Mess := '             ENTER  the right endpoint B = ';
            B := 1;
            WRITE(Mess);
            READLN(B);
          end
        else
          begin
            WRITELN;
            WRITELN('     The  left  endpoint  is  A =', A : 15 : 7);
            WRITELN;
            WRITELN('     The  right endpoint  is  B =', B : 15 : 7);
          end;
        WRITELN;
        WRITE('     Do you want to make a change ? <Y/N> ');
        READLN(Resp);
        WRITELN;
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Change;
            PROBLEM(FunType);
            WRITELN;
            WRITELN('The  current tolerance  is  TOL =', Tol : 15 : 7);
            Mess := 'ENTER  the  NEW  value  of  TOL = ';
            WRITE(Mess);
            READLN(Tol);
            Tol := ABS(Tol);
            if (Tol < Epsilon) then
              Tol := Epsilon;
            WRITELN;
            WRITELN('The current left  endpoint is A =', A : 15 : 7);
            Mess := 'ENTER the  NEW left  endpoint A = ';
            WRITE(Mess);
            READLN(A);
            WRITELN;
            WRITELN('The current right endpoint is B =', B : 15 : 7);
            Mess := 'ENTER the  NEW right endpoint B = ';
            WRITE(Mess);
            READLN(B);
            WRITELN;
          end
        else
          Stat := Done;
      end;
  end;

  procedure RTABLE (R: MATRIX; M: integer);

    var
      I, J, JB, K, KR, Ttype: integer;
      H, Valu: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('     Do you want to see the full Romberg table or only');
    WRITELN;
    WRITELN;
    WRITELN('     the results from the sequential trapezoidal rule ?');
    WRITELN;
    WRITELN;
    WRITELN('     < 1 > The full Romberg table');
    WRITELN;
    WRITELN;
    WRITELN('     < 2 > The sequential trapezoidal rule only');
    WRITELN;
    WRITELN;
    Mess := '     SELECT  < 1 or 2 >  ';
    WRITE(Mess);
    READLN(Ttype);
    if Ttype < 1 then
      Ttype := 1;
    if Ttype > 2 then
      Ttype := 2;
    CLRSCR;
    WRITELN;
    case Ttype of
      1: 
        begin
          WRITELN('     The Romberg integration table is:');
          WRITELN;
          WRITELN('  J           R(J,0)            R(J,1)            R(J,2)            R(J,3)');
          WRITELN;
          if M <= 7 then
            JB := M
          else
            JB := 7;
          for J := 0 to JB do
            begin
              if J <= 3 then
                KR := J
              else
                KR := 3;
              WRITE(J : 3, '   ');
              for K := 0 to KR - 1 do
                WRITE(R[J, K] : 15 : 7, '   ');
              if KR <= 3 then
                WRITE(R[J, KR] : 15 : 7);
              WRITELN;
            end;
          if 4 <= M then
            begin
              WRITELN;
              WRITELN('  J           R(J,4)            R(J,5)            R(J,6)            R(J,7)');
              WRITELN;
              for J := 4 to JB do
                begin
                  if J <= 7 then
                    KR := J
                  else
                    KR := 7;
                  WRITE(J : 3, '   ');
                  for K := 4 to KR - 1 do
                    WRITE(R[J, K] : 15 : 7, '   ');
                  if KR <= 7 then
                    WRITE(R[J, KR] : 15 : 7);
                  WRITELN;
                end;
            end;
        end;
      2: 
        begin
          WRITELN;
          WRITELN('The sequential trapezoidal rule yields:');
          WRITELN;
          WRITELN('  J             T(J) ');
          WRITELN;
          for J := 0 to M do
            begin
              WRITELN(J : 3, '   ', R[J, 0] : 15 : 7);
              WRITELN;
            end;
        end;
    end;
  end;                                           {End procedure RTABLE}

  procedure RESULTS (A, B, Close: real; J, M: integer);
    var
      K, U, V: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN(B : 13 : 5);
    WRITELN('       /');
    WRITE('       |  ');
    PRINTFUNCTION(FunType);
    WRITELN(' DX ~', R[J, J] : 15 : 7);
    WRITELN('       /');
    WRITELN(A : 13 : 5);
    WRITELN;
    WRITELN('The diagonal elements in the  Romberg table  are:');
    if J <= 7 then
      WRITELN;
    for K := 0 to TRUNC(J / 2) do
      begin
        U := 2 * K + 1;
        V := 2 * K + 2;
        if U <= J then
          begin
            WRITE('    R(', U : 1, ',', U : 1, ') =', R[U, U] : 15 : 7, '       ');
            if V <= J then
              WRITELN('R(', V : 1, ',', V : 1, ') =', R[V, V] : 15 : 7)
            else
              WRITELN;
          end;
      end;
    if J <= 9 then
      WRITELN;
    WRITELN(M : 3, ' subintervals were used to compute the approximation for integral of');
    WRITELN;
    WRITE('     F(X) = ');
    PRINTFUNCTION(FunType);
    if J <= 11 then
      WRITELN;
    WRITELN;
    WRITELN('taken over the interval   [', A : 15 : 7, '  ,', B : 15 : 7, '  ].');
    if J <= 5 then
      WRITELN;
    WRITELN('Romberg`s approximation is ', R[J, J] : 15 : 7);
    if J <= 11 then
      WRITELN;
    WRITELN('  The  error  estimate  is ', Close : 15 : 7);
    if Close < Tol then
      WRITELN('It is less the value TOL = ', Tol : 15 : 7)
    else
      WRITELN('  It is LARGER than  TOL = ', Tol : 15 : 7);
  end;

begin                                            {Begin Main Program}
  MESSAGE(Tol);
  DoMo := Go;
  while DoMo = Go do
    begin
      INPUT(A, B);
      State := Working;
      while (State = Working) or (State = Changes) do
        begin
          EPOINTS(A, B, State);
          ROMBERG(A, B, Close, Tol, J, M);
          RESULTS(A, B, Close, J, M);
          WRITELN;
          WRITE('Want to see the divided difference table ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          WRITELN;
          if (Ans = 'y') or (Ans = 'Y') then
            begin
              RTABLE(R, J);
              WRITELN;
              WRITE('Press the <ENTER> key. ');
              READLN(Ans);
              WRITELN;
            end;
          WRITE('Want  to try  another  interval  or  TOL ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans = 'Y') or (Ans = 'y') then
            State := Changes;
          if (Ans <> 'Y') and (Ans <> 'y') then
            State := Done;
        end;
      WRITELN;
      WRITE('Do you want  to use a different function ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        DoMo := Stop;
    end;                                          {End of Main Program}
end.

